// 扩展上下文检查函数
function isExtensionContextValid() {
    try {
        // 尝试访问chrome.runtime.id - 如果扩展上下文无效，这会抛出错误
        return !!chrome.runtime.id;
    } catch (e) {
        console.debug('扩展上下文已失效', e);
        return false;
    }
}

// 安全发送消息
async function sendMessageSafely(tabId, message) {
    if (!isExtensionContextValid()) {
        return { success: false, error: 'Extension context invalidated' };
    }
    
    return new Promise((resolve) => {
        try {
            chrome.tabs.sendMessage(tabId, message, (response) => {
                if (chrome.runtime.lastError) {
                    console.debug('发送消息时出错:', chrome.runtime.lastError.message);
                    resolve({ 
                        success: false, 
                        error: chrome.runtime.lastError.message 
                    });
                } else {
                    resolve(response || { success: true });
                }
            });
        } catch (err) {
            console.error('发送消息时发生异常:', err);
            resolve({ 
                success: false, 
                error: err.message || JSON.stringify(err) 
            });
        }
    });
}

// 处理标签页激活事件
chrome.tabs.onActivated.addListener((activeInfo) => {
    if (!isExtensionContextValid()) return;
    
    // 获取当前设置和高亮数据
    chrome.storage.local.get(['enabled', 'highlights', 'settings'], (data) => {
        if (!isExtensionContextValid()) return;
        
        const settings = data.settings ?? {
            autoHighlight: true,
            caseSensitive: false,
            globalMatch: true,
            onlyCurrentTab: false
        };
        
        // 如果启用了onlyCurrentTab，需要为新激活的标签页应用正确的高亮
        if (settings.onlyCurrentTab) {
            chrome.tabs.get(activeInfo.tabId, (tab) => {
                if (chrome.runtime.lastError || !tab) {
                    console.debug('获取激活标签页信息失败:', chrome.runtime.lastError);
                    return;
                }
                
                // 检查URL是否支持
                if (isIncompatibleWebsite(tab.url)) {
                    console.debug('激活的标签页不支持高亮:', tab.url);
                    return;
                }
                
                // 获取该标签页的高亮词
                const currentTabId = activeInfo.tabId.toString();
                const highlights = data.highlights ?? {};
                highlights.tabs = highlights.tabs ?? {};
                const tabHighlights = highlights.tabs[currentTabId] ?? [];
                
                // 更新currentHighlights为当前标签页的高亮词
                chrome.storage.local.set({ currentHighlights: tabHighlights }, () => {
                    console.debug('标签页切换：已更新currentHighlights为当前标签页高亮词:', tabHighlights.length);
                    
                    // 确保tabs存储结构完整
                    if (!highlights.tabs[currentTabId]) {
                        highlights.tabs[currentTabId] = tabHighlights;
                        chrome.storage.local.set({ highlights }, () => {
                            console.debug('标签页切换：已初始化tabs存储结构');
                        });
                    }
                    
                    // 发送消息到内容脚本应用高亮
                    chrome.tabs.sendMessage(activeInfo.tabId, {
                        action: 'updateHighlights',
                        enabled: data.enabled !== false,
                        force: true,
                        highlights: tabHighlights,
                        settings: {
                            ...settings,
                            onlyCurrentTab: true,
                            autoHighlight: true
                        }
                    }, (response) => {
                        if (chrome.runtime.lastError) {
                            console.debug('发送高亮更新消息失败:', chrome.runtime.lastError.message);
                        } else {
                            console.debug('标签页切换：高亮更新成功');
                        }
                    });
                });
            });
        }
    });
});

// sendMessageSafely函数已在文件开头定义，此处删除重复定义

// Initialize extension state when installed
chrome.runtime.onInstalled.addListener(() => {
    if (!isExtensionContextValid()) return;
    
    chrome.storage.local.set({
        enabled: true,
        highlights: {
            default: [], // 全局高亮词（当onlyCurrentTab为false时使用）
            tabs: {} // 按标签页存储的高亮词（当onlyCurrentTab为true时使用）
        },
        settings: {
            autoHighlight: true, // 默认开启自动高亮
            caseSensitive: false,
            globalMatch: true,
            onlyCurrentTab: false, // 默认关闭只在当前标签页应用
            highlightDelay: 800,
            renderMode: 'stable',
            useHardwareAcceleration: true,
            usePositionLocking: true
        }
    });
});

// Handle messages from content script or popup
chrome.runtime.onMessage.addListener((request, sender, sendResponse) => {
    if (!isExtensionContextValid()) {
        console.debug('接收消息时扩展上下文已失效');
        if (sendResponse) {
            sendResponse({ success: false, error: 'Extension context invalidated' });
        }
        return true;
    }
    
    // 处理内容脚本就绪通知
    if (request.action === 'contentScriptReady') {
        console.debug('收到内容脚本就绪通知', sender.tab?.url);
        sendResponse({ received: true });
        return true;
    }
    
    if (request.action === 'addHighlight') {
        try {
            // 获取当前活动标签页信息
            chrome.tabs.query({active: true, currentWindow: true}, function(activeTabs) {
                if (!activeTabs.length) {
                    if (sendResponse) {
                        sendResponse({ success: false, error: 'No active tab found' });
                    }
                    return;
                }
                
                const activeTabId = activeTabs[0].id.toString();
                
                chrome.storage.local.get(['highlights', 'settings'], function(data) {
                    if (!isExtensionContextValid()) {
                        if (sendResponse) {
                            sendResponse({ success: false, error: 'Extension context invalidated during operation' });
                        }
                        return;
                    }
                    
                    const highlights = data.highlights ?? {};
                    highlights.default = highlights.default ?? [];
                    highlights.tabs = highlights.tabs ?? {};
                    const settings = data.settings ?? {};
                    
                    const newHighlight = {
                        text: request.text,
                        color: request.color ?? '#ffeb3b',
                        id: Date.now().toString(),
                        caseSensitive: settings.caseSensitive,
                        globalMatch: settings.globalMatch
                    };
                    
                    let targetHighlights;
                    let isDuplicate = false;
                    
                    if (settings.onlyCurrentTab) {
                        // 只在当前标签页应用：存储到当前标签页
                        highlights.tabs[activeTabId] = highlights.tabs[activeTabId] ?? [];
                        targetHighlights = highlights.tabs[activeTabId];
                        isDuplicate = targetHighlights.some(h => h.text === newHighlight.text);
                    } else {
                        // 应用到所有标签页：存储到全局
                        targetHighlights = highlights.default;
                        isDuplicate = targetHighlights.some(h => h.text === newHighlight.text);
                    }
                    
                    if (!isDuplicate) {
                        targetHighlights.push(newHighlight);
                        chrome.storage.local.set({ highlights }, () => {
                            if (!isExtensionContextValid()) {
                                if (sendResponse) {
                                    sendResponse({ success: false, error: 'Extension context invalidated during save' });
                                }
                                return;
                            }
                            
                            // 根据onlyCurrentTab设置决定更新范围
                            if (settings.autoHighlight) {
                                if (settings.onlyCurrentTab) {
                                    // 只更新当前活动标签页，使用当前标签页的高亮词
                                    const currentTabHighlights = highlights.tabs[activeTabId] ?? [];
                                    updateTabHighlights(activeTabs[0], currentTabHighlights, settings);
                                } else {
                                    // 更新所有兼容的标签页，使用全局高亮词
                                    chrome.tabs.query({}, async (allTabs) => {
                                        if (!isExtensionContextValid() || !allTabs.length) return;
                                        
                                        for (const tab of allTabs) {
                                            await updateTabHighlights(tab, highlights.default, settings);
                                        }
                                    });
                                }
                            }
                            
                            // 发送响应表示请求已处理
                            if (sendResponse) {
                                sendResponse({ success: true });
                            }
                        });
                    } else {
                        if (sendResponse) {
                            sendResponse({ success: true, existed: true });
                        }
                    }
                });
            });
            });
        } catch (error) {
            console.error('处理addHighlight消息时出错:', error);
            if (sendResponse) {
                sendResponse({ 
                    success: false, 
                    error: error.message || JSON.stringify(error) 
                });
            }
        }
        return true;
    }
    
    // 如果没有处理，也发送响应
    if (sendResponse) {
        sendResponse({ success: false, error: 'Unknown action: ' + request.action });
    }
    return true;
});

// 辅助函数：更新单个标签页的高亮
async function updateTabHighlights(tab, highlights, settings) {
    // 检查是否是有效的标签页
    if (!tab || !tab.id || tab.id === chrome.tabs.TAB_ID_NONE) {
        return;
    }
    
    // 只发送消息到http/https页面
    if (!tab.url || !tab.url.match(/^https?:\/\//)) {
        return;
    }
    
    // 检查是否是不兼容的网站
    if (isIncompatibleWebsite(tab.url)) {
        return;
    }
    
    try {
        await sendMessageSafely(tab.id, {
            action: 'updateHighlights',
            enabled: true,
            force: false,
            highlights: highlights.map(h => ({
                ...h,
                globalMatch: settings.globalMatch !== false,
                processAll: false,
                multipleMatches: true,
                maintainAllHighlights: false
            })),
            settings: {
                ...settings,
                batchSize: 50, // 增加批处理大小以提高性能
                processingInterval: 50, // 减少处理间隔
                processAll: false,
                performanceMode: true
            }
        });
    } catch (error) {
        console.debug('发送消息到标签页出错:', error);
    }
}

// 检查URL是否是不兼容的网站
function isIncompatibleWebsite(url) {
    if (!url || typeof url !== 'string') return true;
    
    // 只处理HTTP/HTTPS页面
    if (!url.match(/^https?:\/\//)) {
        return true;
    }
    
    // 检查不兼容的网站列表
    const incompatibleSites = [
        'feishu.cn/base/',
        'feishu.cn/bitable/',
        'lark.com/base/',
        'lark.com/bitable/',
        'shimo.im/sheet',
        'airtable.com',
        'notion.so',
        'figma.com',
        'canva.com',
        'trello.com',
        'miro.com',
        'github.com/blob',
        'codesandbox.io',
        'codepen.io',
        'jsfiddle.net'
    ];
    
    for (const site of incompatibleSites) {
        if (url.includes(site)) {
            console.debug('检测到不兼容网站:', url);
            return true;
        }
    }
    
    return false;
}

// Listen for tab updates
chrome.tabs.onUpdated.addListener((tabId, changeInfo, tab) => {
    if (!isExtensionContextValid()) return;
    
    if (changeInfo.status === 'complete') {
        // 确保tab存在且有效
        if (!tab || !tab.id || tab.id === chrome.tabs.TAB_ID_NONE) {
            console.debug('Tab无效，无法发送消息');
            return;
        }
        
        // 检查URL是否支持
        if (isIncompatibleWebsite(tab.url)) {
            console.debug('不支持的URL类型或不兼容的网站:', tab.url);
            return;
        }
        
        // 获取当前设置并应用
        chrome.storage.local.get(['enabled', 'highlights', 'settings'], async (data) => {
            if (!isExtensionContextValid()) return;
            
            const settings = data.settings ?? {
                autoHighlight: true,
                caseSensitive: false,
                globalMatch: true,
                onlyCurrentTab: false
            };
            
            // 如果启用了onlyCurrentTab，只处理当前活动标签页
            if (settings.onlyCurrentTab) {
                chrome.tabs.query({active: true, currentWindow: true}, (activeTabs) => {
                    if (!activeTabs.length || activeTabs[0].id !== tabId) {
                        console.debug('onlyCurrentTab启用，跳过非活动标签页:', tabId);
                        return;
                    }
                    // 继续处理当前活动标签页
                    processTabUpdate(tabId, tab, data, settings);
                });
            } else {
                // 处理所有标签页
                processTabUpdate(tabId, tab, data, settings);
            }
        });
    }
});

// 提取标签页更新处理逻辑
function processTabUpdate(tabId, tab, data, settings) {
    if (!isExtensionContextValid()) return;
    
    const finalSettings = {
        autoHighlight: true,
        caseSensitive: false,
        globalMatch: true,
        batchSize: 50, // 增加批处理大小以提高性能
        processingInterval: 50, // 减少处理间隔
        processAll: false,
        multipleMatches: true,
        maintainAllHighlights: false,
        reprocessExisting: false,
        performanceMode: true,
        ...settings
    };
    
    // 首先检查内容脚本是否已经准备好
    sendMessageSafely(tabId, { action: 'ping' })
        .then(pingResult => {
            if (!pingResult.success || !pingResult.scriptReady) {
                console.debug('内容脚本还未准备好，延迟发送消息');
                setTimeout(() => attemptSendMessage(), 800);
                return;
            }
            attemptSendMessage();
        })
        .catch(error => {
            console.debug('检查内容脚本状态时出错:', error);
            setTimeout(() => attemptSendMessage(), 800);
        });
    
    function attemptSendMessage() {
        if (!isExtensionContextValid()) return;
        
        chrome.tabs.get(tabId, async (currentTab) => {
            if (chrome.runtime.lastError || !currentTab) {
                console.debug('获取标签页信息失败:', chrome.runtime.lastError);
                return;
            }
            
            if (isIncompatibleWebsite(currentTab.url)) {
                console.debug('标签页URL已变为不兼容网站，跳过消息发送');
                return;
            }
            
            // 根据onlyCurrentTab设置选择高亮词来源
            let highlightsToUse = [];
            if (finalSettings.onlyCurrentTab) {
                // 使用当前标签页的高亮词
                const tabHighlights = data.highlights?.tabs?.[tabId.toString()] ?? [];
                highlightsToUse = tabHighlights;
            } else {
                // 使用全局高亮词
                highlightsToUse = data.highlights?.default ?? [];
            }
            
            // 发送优化的消息到content script
            const result = await sendMessageSafely(tabId, {
                action: 'updateHighlights',
                enabled: data.enabled ?? true,
                force: false,
                highlights: highlightsToUse.map(h => ({
                     ...h,
                     globalMatch: finalSettings.globalMatch !== false,
                     processAll: false,
                     multipleMatches: true,
                     maintainAllHighlights: false
                 })) ?? [],
                 settings: finalSettings
             });
             
             if (result && result.success) {
                 console.debug('成功发送高亮更新到标签页:', tabId);
             } else {
                 console.debug('发送高亮更新失败:', tabId, result);
             }
         });
     }
 }